//-------------------------------------------------------------------------------------
// SID Monitor - Utility for Sudden Ionospheric Disturbances Monitoring Stations
// Copyright (C) 2005 - Lionel Loudet
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//-------------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.IO;

namespace SID_monitor
{
    public partial class OutputTextBox : UserControl
    {
        public OutputTextBox()
        {
            InitializeComponent();
        }

        #region Output Text Box functions
        //*********************************************************************
        //
        // richTextBox
        //
        //*********************************************************************

        // This delegate enables asynchronous calls for setting the text property
        delegate void AddOutputTextMessageCallback(string text, bool showDate, OutputTextMessageCategory messageCategory);

        public enum OutputTextMessageCategory { Normal, Warning, Error, RRDToolNormal, RRDToolError };

        public void AddOutputTextMessage(string text)
        {
            AddOutputTextMessage(text, true, OutputTextMessageCategory.Normal);
        }

        public void AddOutputTextMessageWithoutDate(string text)
        {
            AddOutputTextMessage(text, false, OutputTextMessageCategory.Normal);
        }

        protected void AddOutputTextMessage(string text, bool showDate, OutputTextMessageCategory messageCategory)
        {
            // InvokeRequired required compares the thread ID of the calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if (this.richTextBox.InvokeRequired)
            {
                AddOutputTextMessageCallback d = new AddOutputTextMessageCallback(AddOutputTextMessage);
                this.Invoke(d, new object[] { text, showDate, messageCategory });
            }
            else
            {
                switch (messageCategory)
                { // sets the text color
                    case OutputTextMessageCategory.Normal:
                        this.richTextBox.SelectionColor = System.Drawing.SystemColors.WindowText;
                        break;
                    case OutputTextMessageCategory.Warning:
                        this.richTextBox.SelectionColor = System.Drawing.Color.DarkOrange;
                        break;
                    case OutputTextMessageCategory.Error:
                        this.richTextBox.SelectionColor = System.Drawing.Color.Red;
                        break;
                    case OutputTextMessageCategory.RRDToolNormal:
                        this.richTextBox.SelectionColor = System.Drawing.Color.Blue;
                        break;
                    case OutputTextMessageCategory.RRDToolError:
                        this.richTextBox.SelectionColor = System.Drawing.Color.Red;
                        break;
                    default:
                        this.richTextBox.SelectionColor = System.Drawing.SystemColors.WindowText;
                        break;
                }

                if (showDate)
                {
                    this.richTextBox.AppendText(DateTime.UtcNow.ToString() + ": " + text);
                }
                else
                {
                    this.richTextBox.AppendText(text);
                }
                this.richTextBox.ScrollToCaret();
                this.richTextBox.Refresh();
            }
        }

        public void AddOutputTextWarningMessage(string text)
        {
            AddOutputTextMessage(text, true, OutputTextMessageCategory.Warning);
        }

        public void AddOutputTextErrorMessage(string text)
        {
            AddOutputTextMessage(text, true, OutputTextMessageCategory.Error);
        }

        public void AddOutputTextRRDToolMessage(string text)
        {
            AddOutputTextMessage(text, true, OutputTextMessageCategory.RRDToolNormal);
        }

        public void AddOutputTextRRDToolErrorMessage(string text)
        {
            AddOutputTextMessage(text, true, OutputTextMessageCategory.RRDToolError);
        }


        /// <summary>
        /// Log File Creation when user right-clicks on the Output Text Box
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void toolStripMenuItemSaveLogFile_Click(object sender, EventArgs e)
        {
            if (saveFileDialogLogFile.ShowDialog() == DialogResult.OK)
            {
                try
                {
                    FileStream fsLog = new FileStream(saveFileDialogLogFile.FileName, FileMode.Create, FileAccess.Write, FileShare.Read);
                    TextWriter wrLog = new StreamWriter(fsLog);
                    try
                    {
                        wrLog.Write(this.richTextBox.Rtf);
                    }
                    catch (Exception ex)
                    {
                        AddOutputTextErrorMessage("*** Cannot write log file. (" + saveFileDialogLogFile.FileName + ")\n" + ex.ToString().Split('\n')[0] + "\n");
                    }
                    finally
                    {
                        wrLog.Close();
                        fsLog.Close();
                    }
                }
                catch (Exception ex)
                {
                    AddOutputTextErrorMessage("*** Cannot write log file. (" + saveFileDialogLogFile.FileName + ")\n" + ex.ToString().Split('\n')[0] + "\n");
                }


            }
        }
        #endregion

    }
}
